/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2006 Phex Development Group
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.common;

import java.lang.reflect.*;

import org.apache.commons.lang.SystemUtils;

import phex.utils.NLogger;
import phex.utils.NLoggerNames;



/**
 * This class assists in tracking Thread use.
 */
public class ThreadTracking
{
    public static ThreadGroup threadPoolGroup;
    public static ThreadGroup rootThreadGroup;

    private static ThreadGroup systemGroup;

    public static void initialize()
    {
        // we want the system thread group
        systemGroup = Thread.currentThread().getThreadGroup();
        while ( systemGroup.getParent() != null )
        {// not the system thread group.. go up one step
            systemGroup = systemGroup.getParent();
        }
        
        if ( SystemUtils.isJavaVersionAtLeast(1.5f) )
        {
            prepare15UncaughtExceptionHandler();
        }
        
        // TODO in the future all thread creation should go through this class
        // to consistently use uncaught exception handling.
        rootThreadGroup = new PhexThreadGroup( "PhexRoot" );
        threadPoolGroup = new PhexThreadGroup( "PhexThreadPool" );
    }
    
    private static void prepare15UncaughtExceptionHandler()
    {
        try
        {
            InvocationHandler invoHandler = new InvocationHandler()
            {
                public Object invoke( Object proxy, Method method, Object[] args ) throws Throwable
                {
                    if ( method != null && "uncaughtException".equals( method.getName() ) )
                    {
                        Thread thread = (Thread) args[0];
                        Throwable throwable = (Throwable) args[1];
                        NLogger.error(NLoggerNames.GLOBAL, 
                            "Uncaught exception: " + throwable.getMessage() + " in Thread: " 
                            + thread.getName(), throwable );
                    }
                    return null;
                }
            };
            Object expHandler = Proxy.newProxyInstance(
                ClassLoader.getSystemClassLoader(),
                new Class[] {Class.forName("java.lang.Thread$UncaughtExceptionHandler")},
                invoHandler );
            Method setMethod = Thread.class.getDeclaredMethod(
                "setDefaultUncaughtExceptionHandler", 
                new Class[] {Class.forName("java.lang.Thread$UncaughtExceptionHandler")} );
            setMethod.invoke( null, new Object[] {expHandler});
        }
        catch (IllegalArgumentException exp)
        {
            NLogger.error( ThreadTracking.class, exp, exp );
        }
        catch (ClassNotFoundException exp)
        {
            NLogger.error( ThreadTracking.class, exp, exp );
        }
        catch (SecurityException exp)
        {
            NLogger.error( ThreadTracking.class, exp, exp );
        }
        catch (NoSuchMethodException exp)
        {
            NLogger.error( ThreadTracking.class, exp, exp );
        }
        catch (IllegalAccessException exp)
        {
            NLogger.error( ThreadTracking.class, exp, exp );
        }
        catch (InvocationTargetException exp)
        {
            NLogger.error( ThreadTracking.class, exp, exp );
        }
    }

    private static class PhexThreadGroup extends ThreadGroup
    {
        public PhexThreadGroup( String name )
        {
            super( systemGroup, name );
            
        }
        
        public void uncaughtException(Thread t, Throwable e)
        {
            super.uncaughtException(t, e);
            NLogger.error(NLoggerNames.GLOBAL, 
                "Uncaught exception: " + e.getMessage() + " in Thread: " 
                + t.getName(), e );
        }
    }

    /*public static void dumpFullThreadLog()
    {
        if ( !Logger.isLevelLogged( Logger.FINEST ) )
        {
            return;
        }
        int count = systemGroup.activeCount();
        Thread[] threads = new Thread[ count ];
        count = systemGroup.enumerate( threads, true );
        Logger.logMessage( Logger.FINEST, Logger.GLOBAL,
            "------------------- Start Full Thread Dump -------------------" );
        for ( int i = 0; i < count; i++ )
        {
            Logger.logMessage( Logger.FINEST, Logger.GLOBAL, threads[ i ].toString() );
        }
        Logger.logMessage( Logger.FINEST, Logger.GLOBAL,
            "-------------------- End Full Thread Dump --------------------" );
    }*/
}